-- ==========================================
-- SPARQL SUM()/AVG() Aggregate Tests
-- ==========================================

-- Test 1: sum/avg of integers
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"20"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"30"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 2: sum/avg of decimals
WITH j (val) AS (
    VALUES
        ('"10.5"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode),
        ('"20.3"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode),
        ('"30.2"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 3: sum/avg of integer + decimal
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"20.5"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode),
        ('"30"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 4: sum/avg of decimal + float
WITH j (val) AS (
    VALUES
        ('"10.5"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode),
        ('"20.3"^^<http://www.w3.org/2001/XMLSchema#float>'::rdfnode),
        ('"5"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 5: sum/avg of integer + decimal + double
WITH j (val) AS (
    VALUES
        ('"10.4"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode),
        ('"20"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"30.4"^^<http://www.w3.org/2001/XMLSchema#double>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 6: sum/avg of single value
WITH j (val) AS (
    VALUES
        ('"42"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 7: sum/avg with integer subtypes (xsd:int, xsd:long)
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#int>'::rdfnode),
        ('"20"^^<http://www.w3.org/2001/XMLSchema#long>'::rdfnode),
        ('"30"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 8: sum/avg with all four numeric types
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"5.5"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode),
        ('"2.5"^^<http://www.w3.org/2001/XMLSchema#float>'::rdfnode),
        ('"3.0"^^<http://www.w3.org/2001/XMLSchema#double>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 9: sum/avg with integer + float (type promotion)
WITH j (val) AS (
    VALUES
        ('"100"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"0.1"^^<http://www.w3.org/2001/XMLSchema#float>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 10: sum/avg with negative integers
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"-5"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"-3"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 11: sum/avg resulting in negative value
WITH j (val) AS (
    VALUES
        ('"5"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"-20"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 12: sum/avg with zero values
WITH j (val) AS (
    VALUES
        ('"0"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"0"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 13: sum/avg resulting in zero
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"-10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 14: sum/avg with NULL values
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        (NULL::rdfnode),
        ('"30"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 15: sum/avg of all NULLs
WITH j (val) AS (
    VALUES
        (NULL::rdfnode),
        (NULL::rdfnode),
        (NULL::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 16: sum/avg of empty set (no rows)
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j WHERE val IS NULL;

-- Test 17: sum/avg with xsd:string mixed in
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"20"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"30"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 18: sum/avg with non-numeric only 
WITH j (val) AS (
    VALUES
        ('"not-a-number"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 19: sum/avg on multiple xsd:string
WITH j (val) AS (
    VALUES
        ('"apple"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"banana"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.avg(val), sparql.sum(val) FROM j;

-- Test 20: sum/avg on xsd:date
WITH j (val) AS (
    VALUES
        ('"2023-01-01"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode),
        ('"2023-06-15"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 21: sum/avg on xsd:dateTime 
WITH j (val) AS (
    VALUES
        ('"2023-01-01T10:30:00"^^<http://www.w3.org/2001/XMLSchema#dateTime>'::rdfnode),
        ('"2023-06-15T15:45:00"^^<http://www.w3.org/2001/XMLSchema#dateTime>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 22: sum/avg on xsd:time 
WITH j (val) AS (
    VALUES
        ('"10:30:00"^^<http://www.w3.org/2001/XMLSchema#time>'::rdfnode),
        ('"15:45:00"^^<http://www.w3.org/2001/XMLSchema#time>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 23: sum/avg on xsd:duration 
WITH j (val) AS (
    VALUES
        ('"P1Y"^^<http://www.w3.org/2001/XMLSchema#duration>'::rdfnode),
        ('"P2M"^^<http://www.w3.org/2001/XMLSchema#duration>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 24: sum/avg on xsd:boolean
WITH j (val) AS (
    VALUES
        ('"true"^^<http://www.w3.org/2001/XMLSchema#boolean>'::rdfnode),
        ('"false"^^<http://www.w3.org/2001/XMLSchema#boolean>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 25: sum/avg with mixed numeric and xsd:date
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"2023-01-01"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 26: sum/avg with mixed numeric and xsd:string
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"text"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- ==========================================
-- SPARQL MIN/MAX() Aggregate Tests
-- ==========================================

-- Test 27: min/max of integers
WITH j (val) AS (
    VALUES
        ('"30"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"20"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 28: min/max of decimals
WITH j (val) AS (
    VALUES
        ('"30.2"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode),
        ('"10.5"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode),
        ('"20.3"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 29: min/max of integer + decimal
WITH j (val) AS (
    VALUES
        ('"30"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"10.5"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode),
        ('"20"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 30: min/max with float values
WITH j (val) AS (
    VALUES
        ('"30.5"^^<http://www.w3.org/2001/XMLSchema#float>'::rdfnode),
        ('"10.1"^^<http://www.w3.org/2001/XMLSchema#float>'::rdfnode),
        ('"20.3"^^<http://www.w3.org/2001/XMLSchema#float>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 31: min/max with mixed types
WITH j (val) AS (
    VALUES
        ('"30"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"5.5"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode),
        ('"10.0"^^<http://www.w3.org/2001/XMLSchema#float>'::rdfnode),
        ('"20.0"^^<http://www.w3.org/2001/XMLSchema#double>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 32: min/max of single value
WITH j (val) AS (
    VALUES
        ('"42"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 33: min/max with negative integers
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"-5"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"3"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 34: min/max with all negative values
WITH j (val) AS (
    VALUES
        ('"-10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"-5"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"-20"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 35: min/max with zero values
WITH j (val) AS (
    VALUES
        ('"0"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"-5"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 36: min/max with NULL values
WITH j (val) AS (
    VALUES
        ('"30"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        (NULL::rdfnode),
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        (NULL::rdfnode),
        ('"20"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 37: min/max of all NULLs
WITH j (val) AS (
    VALUES
        (NULL::rdfnode),
        (NULL::rdfnode),
        (NULL::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 38: min/max of empty set (no rows)
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j WHERE val IS NULL;

-- Test 39: min/max with equal values
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"10.0"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode),
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 40: min/max on multiple strings
WITH j (val) AS (
    VALUES
        ('"zebra"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"apple"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"mango"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 41: min/max on dates
WITH j (val) AS (
    VALUES
        ('"2023-06-15"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode),
        ('"2021-01-01"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode),
        ('"2025-12-31"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 42: min/max on single non-numeric value → should return that value
WITH j (val) AS (
    VALUES
        ('"single-string"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 43: min/max with numeric + string
WITH j (val) AS (
    VALUES
        ('"zebra"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"42"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"mango"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 44: min/max with numeric + date
WITH j (val) AS (
    VALUES
        ('"2023-06-15"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode),
        ('"100"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"2021-01-01"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 45: min/max with date + string
WITH j (val) AS (
    VALUES
        ('"apple"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"2023-06-15"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode),
        ('"zebra"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 46: min/max with numeric + dateTime
WITH j (val) AS (
    VALUES
        ('"2023-06-15T10:30:00"^^<http://www.w3.org/2001/XMLSchema#dateTime>'::rdfnode),
        ('"5"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"2021-01-01T00:00:00"^^<http://www.w3.org/2001/XMLSchema#dateTime>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 47: min/max with multiple category types
WITH j (val) AS (
    VALUES
        ('"zebra"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"42"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"2023-06-15"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode),
        ('"10:30:00"^^<http://www.w3.org/2001/XMLSchema#time>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 48: min/max with IRI + numeric + string
WITH j (val) AS (
    VALUES
        ('"zebra"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"42"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('<http://example.org/resource/1>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 49: min/max with lang-tagged + numeric
WITH j (val) AS (
    VALUES
        ('"42"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"hello"@en'::rdfnode),
        ('"100"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 50: min/max with plain literal + numeric
WITH j (val) AS (
    VALUES
        ('"42"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"plain-text"'::rdfnode),
        ('"100"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 51: min/max with time + duration
WITH j (val) AS (
    VALUES
        ('"P1Y"^^<http://www.w3.org/2001/XMLSchema#duration>'::rdfnode),
        ('"10:30:00"^^<http://www.w3.org/2001/XMLSchema#time>'::rdfnode),
        ('"P2M"^^<http://www.w3.org/2001/XMLSchema#duration>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 52: min/max with comprehensive mix (IRI, lang, plain, numeric, temporal, string)
WITH j (val) AS (
    VALUES
        ('"string-value"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('<http://example.org/id>'::rdfnode),
        ('"42"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"labeled"@en'::rdfnode),
        ('"plain"'::rdfnode),
        ('"2023-06-15"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 53: min/max with multiple numerics of different types in mix
WITH j (val) AS (
    VALUES
        ('"zebra"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"5.5"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode),
        ('"100"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"2023-01-01"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 54: min/max with multiple temporals of different types
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"2023-06-15"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode),
        ('"2025-12-31"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode),
        ('"2021-01-01T00:00:00"^^<http://www.w3.org/2001/XMLSchema#dateTime>'::rdfnode),
        ('"apple"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 55: min/max with NULL in mixed-type set
WITH j (val) AS (
    VALUES
        ('"zebra"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"42"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        (NULL::rdfnode),
        ('"2023-06-15"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- ==========================================
-- SPARQL GROUP_CONCAT() Aggregate Tests
-- ==========================================

-- Test 56: GROUP_CONCAT with default separator (space)
WITH j (val) AS (
    VALUES
        ('"apple"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"banana"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"cherry"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.group_concat(val, ' ') FROM j;

-- Test 57: GROUP_CONCAT with custom separator (comma)
WITH j (val) AS (
    VALUES
        ('"apple"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"banana"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"cherry"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.group_concat(val, ', ') FROM j;

-- Test 58: GROUP_CONCAT with integers
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"20"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"30"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.group_concat(val, '-') FROM j;

-- Test 59: GROUP_CONCAT with mixed types
WITH j (val) AS (
    VALUES
        ('"42"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"hello"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"3.14"^^<http://www.w3.org/2001/XMLSchema#float>'::rdfnode)
)
SELECT sparql.group_concat(val, ' | ') FROM j;

-- Test 60: GROUP_CONCAT with IRIs
WITH j (val) AS (
    VALUES
        ('<http://example.org/resource1>'::rdfnode),
        ('<http://example.org/resource2>'::rdfnode),
        ('<http://example.org/resource3>'::rdfnode)
)
SELECT sparql.group_concat(val, '; ') FROM j;

-- Test 61: GROUP_CONCAT with language-tagged strings
WITH j (val) AS (
    VALUES
        ('"hello"@en'::rdfnode),
        ('"bonjour"@fr'::rdfnode),
        ('"hola"@es'::rdfnode)
)
SELECT sparql.group_concat(val, ', ') FROM j;

-- Test 62: GROUP_CONCAT with NULL value
WITH j (val) AS (
    VALUES
        ('"apple"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        (NULL::rdfnode),
        ('"cherry"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.group_concat(val, ', ') FROM j;

-- Test 63: GROUP_CONCAT with all NULL
WITH j (val) AS (
    VALUES
        (NULL::rdfnode),
        (NULL::rdfnode),
        (NULL::rdfnode)
)
SELECT sparql.group_concat(val, ', ') FROM j;

-- Test 64: GROUP_CONCAT with empty set
SELECT sparql.group_concat(val, ', ') 
FROM (SELECT NULL::rdfnode AS val WHERE false) AS j;

-- Test 65: GROUP_CONCAT with single value
WITH j (val) AS (
    VALUES
        ('"only-one"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.group_concat(val, ', ') FROM j;

-- Test 66: GROUP_CONCAT with dates
WITH j (val) AS (
    VALUES
        ('"2024-01-15"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode),
        ('"2024-02-20"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode),
        ('"2024-03-25"^^<http://www.w3.org/2001/XMLSchema#date>'::rdfnode)
)
SELECT sparql.group_concat(val, ' to ') FROM j;

-- Test 67: GROUP_CONCAT with booleans
WITH j (val) AS (
    VALUES
        ('"true"^^<http://www.w3.org/2001/XMLSchema#boolean>'::rdfnode),
        ('"false"^^<http://www.w3.org/2001/XMLSchema#boolean>'::rdfnode)
)
SELECT sparql.group_concat(val, ', ') FROM j;

-- Test 68: GROUP_CONCAT with empty string separator
WITH j (val) AS (
    VALUES
        ('"a"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"b"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"c"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.group_concat(val, '') FROM j;

-- Test 69: GROUP_CONCAT with newline separator
WITH j (val) AS (
    VALUES
        ('"line1"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"line2"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"line3"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.group_concat(val, E'\n') FROM j;

-- Test 70: GROUP_CONCAT with very long separator
WITH j (val) AS (
    VALUES
        ('"first"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"second"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.group_concat(val, ' --- SEPARATOR --- ') FROM j;

-- Test 71: GROUP_CONCAT with decimals
WITH j (val) AS (
    VALUES
        ('"10.50"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode),
        ('"20.30"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode),
        ('"30.20"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode)
)
SELECT sparql.group_concat(val, ', ') FROM j;

-- ==========================================
-- SPARQL SAMPLE() Aggregate Tests
-- ==========================================

-- Test 72: SAMPLE with single integer value
WITH j (val) AS (
    VALUES
        ('"42"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sample(val) FROM j;

-- Test 73: SAMPLE with multiple integers
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"20"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"30"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sample(val) FROM j;

-- Test 74: SAMPLE with NULLs at start
WITH j (val) AS (
    VALUES
        (NULL::rdfnode),
        (NULL::rdfnode),
        ('"42"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"99"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sample(val) FROM j;

-- Test 75: SAMPLE with all NULLs
WITH j (val) AS (
    VALUES
        (NULL::rdfnode),
        (NULL::rdfnode),
        (NULL::rdfnode)
)
SELECT sparql.sample(val) FROM j;

-- Test 76: SAMPLE with empty set
WITH j (val) AS (
    SELECT NULL::rdfnode WHERE false
)
SELECT sparql.sample(val) FROM j;

-- Test 77: SAMPLE with mixed numeric types
WITH j (val) AS (
    VALUES
        ('"10.5"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode),
        ('"20"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"30.7"^^<http://www.w3.org/2001/XMLSchema#float>'::rdfnode)
)
SELECT sparql.sample(val) FROM j;

-- Test 78: SAMPLE with strings
WITH j (val) AS (
    VALUES
        ('"apple"'::rdfnode),
        ('"banana"'::rdfnode),
        ('"cherry"'::rdfnode)
)
SELECT sparql.sample(val) FROM j;

-- Test 79: SAMPLE with IRIs
WITH j (val) AS (
    VALUES
        ('<http://example.org/resource1>'::rdfnode),
        ('<http://example.org/resource2>'::rdfnode),
        ('<http://example.org/resource3>'::rdfnode)
)
SELECT sparql.sample(val) FROM j;

-- Test 80: SAMPLE with language-tagged literals
WITH j (val) AS (
    VALUES
        ('"hello"@en'::rdfnode),
        ('"bonjour"@fr'::rdfnode),
        ('"hola"@es'::rdfnode)
)
SELECT sparql.sample(val) FROM j;

-- Test 81: SAMPLE with mixed types (IRI, literal, numeric) → should return first value
WITH j (val) AS (
    VALUES
        ('<http://example.org/resource1>'::rdfnode),
        ('"some text"'::rdfnode),
        ('"42"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"hello"@en'::rdfnode)
)
SELECT sparql.sample(val) FROM j;

/* == Corner Cases == */

-- Test 82: sum/avg with very large integers near PostgreSQL numeric limits
WITH j (val) AS (
    VALUES
        ('"999999999999999999999999999"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"1"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 83: sum/avg with infinity
WITH j (val) AS (
    VALUES
        ('"INF"^^<http://www.w3.org/2001/XMLSchema#double>'::rdfnode),
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 84: sum/avg with NaN
WITH j (val) AS (
    VALUES
        ('"NaN"^^<http://www.w3.org/2001/XMLSchema#double>'::rdfnode),
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.sum(val), sparql.avg(val) FROM j;

-- Test 85: min/max with blank nodes
WITH j (val) AS (
    VALUES
        ('_:b1'::rdfnode),
        ('_:b2'::rdfnode),
        ('"42"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 86: min/max with same text, different languages
WITH j (val) AS (
    VALUES
        ('"hello"@en'::rdfnode),
        ('"hello"@en-US'::rdfnode),
        ('"hello"@en-GB'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 87: GROUP_CONCAT with quotes, separators in values
WITH j (val) AS (
    VALUES
        ('"text with ""quotes"""^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"text, with, commas"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.group_concat(val, ', ') FROM j;

-- Test 88: min/max when 10 (int) vs 10.0 (decimal) vs 10.0 (float)
WITH j (val) AS (
    VALUES
        ('"10"^^<http://www.w3.org/2001/XMLSchema#integer>'::rdfnode),
        ('"10.0"^^<http://www.w3.org/2001/XMLSchema#decimal>'::rdfnode),
        ('"10.0"^^<http://www.w3.org/2001/XMLSchema#float>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;

-- Test 89: min/max with empty strings and whitespace
WITH j (val) AS (
    VALUES
        ('""^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('" "^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode),
        ('"a"^^<http://www.w3.org/2001/XMLSchema#string>'::rdfnode)
)
SELECT sparql.min(val), sparql.max(val) FROM j;