# pg_isok  A query centered monitoring tool for PostgreSQL
# Copyright (C) 2012, 2014, 2015, 2025 The Meme Factory, Inc.,
#               http://www.karlpinc.com/
# Copyright (C) 2005-2011 Karl O. Pinc <kop@karlpinc.com>
#
#  This program is free software: you can redistribute it and/or modify
#  it under the terms of the GNU Affero General Public License as published
#  by the Free Software Foundation, either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU Affero General Public License for more details.
#
#  You should have received a copy of the GNU Affero General Public License
#  along with this program.  If not, see <https://www.gnu.org/licenses/>.
#
# Makefile for documentation
#
# Karl O. Pinc <kop@karlpinc.com>
#
# Bugs:
#

##
## ##################################################################
##                        Documentation targets
##
## Documentation is built and put into docs_built/.
##
## To generate all the documentation use the "docs" target,
## invoked with `make docs`.
##

# Where the documentation source dir is
DOC_PATH := doc_src
# Where constructed documentation goes
BUILT_PATH := doc
# Where constructed HTML documentation goes, along with everything needed
# to support it.
BUILT_HTML_PATH := $(BUILT_PATH)/pg_isok_html


##   docs
##       Generate all the documentation
DOC_TARGETS := paginated_html_docs $(BUILT_HTML_PATH)/isok.html \
               $(BUILT_PATH)/pg_isok.txt $(BUILT_PATH)/index.html \
               $(BUILT_PATH)/pg_isok_usletter.pdf $(BUILT_PATH)/pg_isok_a4.pdf

docs: $(DOC_TARGETS)

# Add the docs to the "all" target:
all: docs

#
# Other variables
#

# The dependencies causing everything to be rebuilt
MAKEDEPENDS_DOC := $(MAKEDEPENDS) \
                   $(DOC_PATH)/xmllint-wrapper $(DOC_PATH)/dblatex-wrapper

# Command used to make pdfs.
DBLATEX := sh <$(DOC_PATH)/dblatex-wrapper $(DOC_PATH)/dblatex-wrapper

# XMLLINT command used to validate xml
export XMLLINT = sh <$(DOC_PATH)/xmllint-wrapper $(DOC_PATH)/xmllint-wrapper


# Command to munge xhtml into txt.
LINKS = links -no-connect -dump
# Command for xslt transformation
XSLTPROC = xsltproc --nonet --xinclude

##
## Other targets are:
##

#
# Diagrams
#

# This defines some variables, so must be included "first", before the
# rules that use the variables are encountered.  But also uses some
# of the previously defined variables, so must be included _after_
# these.
include $(MAKE_FILES)/diagrams.mk

#
# XML validation
#
# Note that we touch the validated files to update the timestamp
# to post-date the timestamp of the validation breadcrumb file.
# This also serves to update the timestamp to post-date all the
# various Makefiles and so forth that building depends on.

# The directory holding validation breadcrumbs
VALIDATE_PATH := $(DOC_PATH)/validate

$(VALIDATE_PATH):
	if [ ! -d $@ ] ; then mkdir $@ ; fi

##   validate_xml
##       Checks all the xml for errors
.PHONY: validate_xml
validate_xml: $(VALIDATE_PATH)/isok

# doc/validate/isok
#   Checks isok.xml, and all the xml it uses, for errors
#   Depend on isok.xml, and the validation of all related
#   xml files.  Depending on validation depends on the xml source file.
$(VALIDATE_PATH)/isok: \
               $(DOC_PATH)/isok.xml \
               $(VALIDATE_PATH)/isok_main_tables \
               $(VALIDATE_PATH)/isok_support_tables \
               $(VALIDATE_PATH)/isok_diagrams \
               $(VALIDATE_PATH)/agpl-3.0 \
               $(VALIDATE_PATH)/cc0-1.0-universal \
               $(MAKEDEPENDS_DOC) \
               | $(VALIDATE_PATH)
	$(XMLLINT) $(DOC_PATH)/isok.xml
	touch $@
#	# RH systems need explicit path to xml.dcl
#	if [ -e /usr/share/sgml/xml.dcl ] ; then \
#	  xmldcl=/usr/share/sgml/xml.dcl ; \
#	fi ; \
#	onsgmls -s $$xmldcl $(DOC_PATH)/isok.xml

$(VALIDATE_PATH)/isok_support_tables: \
                  $(DOC_PATH)/isok_support_tables.xml \
                  $(MAKEDEPENDS_DOC) \
                  | $(VALIDATE_PATH)
	$(XMLLINT) $(DOC_PATH)/isok_support_tables.xml
	touch $@

$(VALIDATE_PATH)/isok_main_tables: \
                  $(DOC_PATH)/isok_main_tables.xml \
                  $(MAKEDEPENDS_DOC) \
                  | $(VALIDATE_PATH)
	$(XMLLINT) $(DOC_PATH)/isok_main_tables.xml
	touch $@

$(VALIDATE_PATH)/isok_diagrams: \
                  $(DOC_PATH)/isok_diagrams.xml \
                  $(MAKEDEPENDS_DOC) \
                  | $(VALIDATE_PATH)
	$(XMLLINT) $(DOC_PATH)/isok_diagrams.xml
	touch $@

$(VALIDATE_PATH)/agpl-3.0: \
                  $(DOC_PATH)/agpl-3.0.dbk \
                  $(MAKEDEPENDS_DOC) \
                  | $(VALIDATE_PATH)
	$(XMLLINT) $(DOC_PATH)/agpl-3.0.dbk
	touch $@

$(VALIDATE_PATH)/cc0-1.0-universal: \
                  $(DOC_PATH)/cc0-1.0-universal.xml \
                  $(MAKEDEPENDS_DOC) \
                  | $(VALIDATE_PATH)
	$(XMLLINT) $(DOC_PATH)/cc0-1.0-universal.xml
	touch $@

#
# Style sheet dependencies
# Update the timestamp of the including file to match that of the included
# file (which is the dependency here in the Makefile).
#

$(DOC_PATH)/isok_cust.xsl: $(DOC_PATH)/isok.xsl
	touch --reference $< $@

$(DOC_PATH)/isok_nochunk_cust.xsl: $(DOC_PATH)/isok.xsl
	touch --reference $< $@

$(DOC_PATH)/pdf_cust_style.xsl: $(DOC_PATH)/pdf_style.xsl
	touch --reference $< $@

$(DOC_PATH)/pdf_style.xsl: $(DOC_PATH)/isok.xsl
	touch --reference $< $@

#
# Static documentation home page for "codeberg pages"
#
$(BUILT_PATH)/index.html: $(DOC_PATH)/index.html
	cp $< $@

#
# Documentation output
#
$(BUILT_PATH): $(MAKEDEPENDS_DOC)
	mkdir -p $(BUILT_PATH)

$(BUILT_HTML_PATH):
	mkdir -p $(BUILT_HTML_PATH)

#
# PDF output
#

# Unfortunately docbook does not seem to generate diagrams
# from eps or svg.
# olinkpdfdb.xml generates isokpdftargets.db and xsltproc only needs the
# name of olinkpdfdb.xml to know this, but actually uses isokpdftargets.db.
#isok.fo: $(VALIDATE_PATH)/isok \
#         $(DOC_PATH)/isokpdftargets.db $(DOC_PATH)/pdf_cust_style.xsl \
#         $(MAKEDEPENDS_DOC) $(DOC_PATH)/entities
#	$(XSLTPROC) \
#	         -o isok.fo \
#	         --stringparam target.database.document \
#	                       "$(DOC_PATH)/olinkpdfdb.xml" \
#	         --stringparam current.docid "isok" \
#	         --param use.extensions '1' \
#	         --param passivtex.extensions '1' \
#	         --param tablecolumns.extension '0' \
#	         $(DOC_PATH)/pdf_cust_style.xsl $(DOC_PATH)/isok.xml

# Command to use to generate PDF documentation
DBLATEX_PDF :=$(DBLATEX) --pdf \
	        --param role=fo \
	        --param target.database.document=olinkpdfdb.xml \
	        --param current.docid=isok \
	        --param doc.collab.show=0 \
	        --param latex.output.revhistory=0 \
	        --param figure.caution='' \
	        --param figure.important='' \
	        --param figure.note='' \
	        --param figure.tip='' \
	        --param figure.warning=''

# All the pre-requisites for making pdf targets
PDF_PREREQS := $(VALIDATE_PATH)/isok \
               $(DOC_PATH)/isokpdftargets.db $(DOC_PATH)/pdf_cust_style.xsl \
	       $(DOC_PATH)/agplpdftargets.db $(DOC_PATH)/cc0pdftargets.db \
               $(DOC_PATH)/entities \
               $(PNG_TARGETS) $(MAKEDEPENDS_DOC) \

##   docs_built/pg_isok_a4.pdf
##       Make an A4 PDF of the documentation
$(BUILT_PATH)/pg_isok_a4.pdf: $(PDF_PREREQS) | $(BUILT_PATH)
	$(DBLATEX_PDF) --param paper.type=a4paper -o $@ \
	               --param current.docid=isok \
	               --param targets.filename=$(DOC_PATH)/isokpdftargets.db \
	               $(DOC_PATH)/isok.xml

##   docs_built/pg_isok_usletter.pdf
##       Make a USletter PDF of the documentation
$(BUILT_PATH)/pg_isok_usletter.pdf: $(PDF_PREREQS) | $(BUILT_PATH)
	$(DBLATEX_PDF) --param paper.type=letterpaper -o $@ \
	               --param current.docid=isok \
	               --param targets.filename=$(DOC_PATH)/isokpdftargets.db \
	               $(DOC_PATH)/isok.xml

#
# TXT output
#

##   docs_built/pg_isok.txt
##       Make the documentation as plain text

$(BUILT_PATH)/pg_isok.txt: $(BUILT_HTML_PATH)/isok.html $(MAKEDEPENDS_DOC)
	$(LINKS) $(BUILT_HTML_PATH)/isok.html > $@

#
# HTML output
#

# The directory hierarchy of docbook images
DOCBOOK_IMAGES_PATH := $(DOC_PATH)/docbook_html_images
DOCBOOK_IMAGES := $(wildcard $(DOCBOOK_IMAGES_PATH)/*.*) \
                  $(wildcard $(DOCBOOK_IMAGES_PATH)/callouts/*)

HTML_IMAGES_PATH := $(BUILT_HTML_PATH)/images
HTML_IMAGES := $(patsubst \
       $(DOCBOOK_IMAGES_PATH)/%,$(HTML_IMAGES_PATH)/%,$(DOCBOOK_IMAGES))
HTML_IMAGES_DIRS := $(BUILT_HTML_PATH)/images/callouts

$(HTML_IMAGES_DIRS):
	mkdir -p $(HTML_IMAGES_DIRS)

$(HTML_IMAGES): $(HTML_IMAGES_PATH)/%: $(DOCBOOK_IMAGES_PATH)/% \
                $(MAKEDEPENDS_DOC) \
                | $(HTML_IMAGES_DIRS)
	cp $< $@

# All the pre-requisites for making HTML docs
HTML_PREREQS := $(VALIDATE_PATH)/isok \
                $(DOC_PATH)/entities \
                $(SVG_TARGETS) \
                $(MAKEDEPENDS_DOC) \
                $(HTML_IMAGES) \
                $(BUILT_HTML_PATH)/isok.css

##   paginated_html_docs
##       Makes a html_paginated dir of html docs
##       (One page per top-level section)
.PHONY: paginated_html_docs
paginated_html_docs: $(DOC_PATH)/html_paginated/made $(MAKEDEPENDS_DOC)

# Directory in which to put breadcrumbs regards chunked html output
$(DOC_PATH)/html_paginated: $(MAKEDEPENDS_DOC)
	mkdir -p $(DOC_PATH)/html_paginated
	touch $@

# Leave breadcrumb to note that the complete directory of html files is
# up-to-date.
$(DOC_PATH)/html_paginated/made: $(BUILT_HTML_PATH)/html_paginated/index.html \
                            | $(DOC_PATH)/html_paginated
	touch $@

# Actually make the paginated html docs
#    Generate tables of contents in level 1 sections.
#    (The same as our chunking.)
# olinkdb.xml generates isoktargets.db and xsltproc only needs the
# name of olinkdb.xml to know this, but actually uses isoktargets.db.
$(BUILT_HTML_PATH)/html_paginated/index.html: \
                $(HTML_PREREQS) \
                $(DOC_PATH)/isoktargets.db $(DOC_PATH)/isok_cust.xsl \
	        $(DOC_PATH)/agpltargets.db $(DOC_PATH)/cc0targets.db \
                $(MAKEDEPENDS_DOC) | $(BUILT_HTML_PATH)

	mkdir -p $(dir $@) \
	&& $(XSLTPROC) \
	         -o $(dir $@)/ \
	         --stringparam target.database.document "olinkdb.xml" \
	         --stringparam current.docid "isok" \
	         --stringparam chunker.output.indent "yes" \
	         --stringparam profile.condition "html_paginated" \
	         --stringparam html.stylesheet "../isok.css" \
	         --stringparam callout.graphics.path "../images/callouts/" \
	         --stringparam admon.graphics 1 \
	         --stringparam admon.graphics.path "../images/" \
	         $(DOC_PATH)/isok_cust.xsl $(DOC_PATH)/isok.xml

##   doc/pg_isok_html/isok.html
##       Makes a html version of the docs
##       (One big page)
# olinkdb_nochunk.xml generates isoknochunktargets.db and xsltproc
# only needs the name of olinkdb_nochunk.xml to know this, but
# actually uses isoknochunktargets.db. 
$(BUILT_HTML_PATH)/isok.html: $(HTML_PREREQS) \
           $(DOC_PATH)/isoknochunktargets.db \
           $(DOC_PATH)/agplnochunktargets.db \
           $(DOC_PATH)/cc0nochunktargets.db \
           $(DOC_PATH)/isok.xsl | $(BUILT_HTML_PATH)

	$(XSLTPROC) \
	         -o $@ \
	         --stringparam target.database.document "olinkdb_nochunk.xml" \
	         --stringparam current.docid "isok" \
	         --stringparam profile.condition "html_single_page" \
	         --stringparam html.stylesheet "isok.css" \
	         --stringparam admon.graphics 1 \
	         $(DOC_PATH)/isok_nochunk_cust.xsl $(DOC_PATH)/isok.xml

#
# CSS
#
$(BUILT_HTML_PATH)/isok.css: $(DOC_PATH)/isok.css $(MAKEDEPENDS_DOC)
	cp $(DOC_PATH)/isok.css $(BUILT_HTML_PATH)/

#
# The olink cross reference databases
#

$(DOC_PATH)/isoktargets.db: $(VALIDATE_PATH)/isok $(DOC_PATH)/isok_cust.xsl \
              $(DOC_PATH)/olinkdb.xml \
              $(MAKEDEPENDS_DOC)
	$(XSLTPROC) \
	         --stringparam collect.xref.targets "only" \
	         --stringparam target.database.document \
	                       "$(DOC_PATH)/olinkdb.xml" \
	         --stringparam targets.filename "$(DOC_PATH)/isoktargets.db" \
	         $(DOC_PATH)/isok_cust.xsl \
	         $(DOC_PATH)/isok.xml

$(DOC_PATH)/agpltargets.db: $(VALIDATE_PATH)/agpl-3.0 \
              $(DOC_PATH)/isok_cust.xsl \
              $(DOC_PATH)/olinkdb.xml \
              $(MAKEDEPENDS_DOC)
	$(XSLTPROC) \
	         --stringparam collect.xref.targets "only" \
	         --stringparam target.database.document \
	                       "$(DOC_PATH)/olinkdb.xml" \
	         --stringparam targets.filename "$(DOC_PATH)/agpltargets.db" \
	         --stringparam current.docid agpl \
	         --stringparam root.filename '' \
	         $(DOC_PATH)/isok_cust.xsl \
	         $(DOC_PATH)/agpl-3.0.dbk

$(DOC_PATH)/cc0targets.db: $(VALIDATE_PATH)/cc0-1.0-universal \
              $(DOC_PATH)/isok_cust.xsl \
              $(DOC_PATH)/olinkdb.xml \
              $(MAKEDEPENDS_DOC)
	$(XSLTPROC) \
	         --stringparam collect.xref.targets "only" \
	         --stringparam target.database.document \
	                       "$(DOC_PATH)/olinkdb.xml" \
	         --stringparam targets.filename "$(DOC_PATH)/cc0targets.db" \
	         --stringparam current.docid cc0 \
	         --stringparam root.filename '' \
	         $(DOC_PATH)/isok_cust.xsl \
	         $(DOC_PATH)/cc0-1.0-universal.xml

# This just flat out does not work.  We don't care because we
# can use the xhtml xsl to generate the db because the names are
# all the same.
$(DOC_PATH)/isokpdftargets.db: $(VALIDATE_PATH)/isok \
                 $(DOC_PATH)/pdf_cust_style.xsl \
                 $(DOC_PATH)/olinkpdfdb.xml \
                 $(MAKEDEPENDS_DOC)
	$(XSLTPROC) \
	         --stringparam collect.xref.targets "only" \
	         --stringparam target.database.document "olinkpdfdb.xml" \
	         --stringparam targets.filename \
	                       "$(DOC_PATH)/isokpdftargets.db" \
	         $(DOC_PATH)/isok_cust.xsl \
	         $(DOC_PATH)/isok.xml

$(DOC_PATH)/agplpdftargets.db: $(VALIDATE_PATH)/agpl-3.0 \
                 $(DOC_PATH)/pdf_cust_style.xsl \
                 $(DOC_PATH)/olinkpdfdb.xml \
                 $(MAKEDEPENDS_DOC)
	$(XSLTPROC) \
	         --stringparam collect.xref.targets "only" \
	         --stringparam target.database.document "olinkpdfdb.xml" \
	         --stringparam targets.filename \
	                       "$(DOC_PATH)/agplpdftargets.db" \
	         $(DOC_PATH)/isok_cust.xsl \
	         $(DOC_PATH)/agpl-3.0.dbk

$(DOC_PATH)/cc0pdftargets.db: $(VALIDATE_PATH)/cc0-1.0-universal \
                 $(DOC_PATH)/pdf_cust_style.xsl \
                 $(DOC_PATH)/olinkpdfdb.xml \
                 $(MAKEDEPENDS_DOC)
	$(XSLTPROC) \
	         --stringparam collect.xref.targets "only" \
	         --stringparam target.database.document "olinkpdfdb.xml" \
	         --stringparam targets.filename \
	                       "$(DOC_PATH)/cc0pdftargets.db" \
	         $(DOC_PATH)/isok_cust.xsl \
	         $(DOC_PATH)/cc0-1.0-universal.xml

#	xsltproc --nonet \
#	         --stringparam collect.xref.targets "only" \
#	         --stringparam target.database.document "olinkpdfdb.xml" \
#	         --stringparam targets.filename \
#	                       "$(DOC_PATH)/isokpdftargets.db" \
#	         $(DOC_PATH)/pdf_cust_style.xsl \
#	         $(DOC_PATH)/isok.xml

$(DOC_PATH)/isoknochunktargets.db: $(VALIDATE_PATH)/isok \
                     $(DOC_PATH)/isok_nochunk_cust.xsl \
                     $(DOC_PATH)/olinkdb_nochunk.xml \
                     $(MAKEDEPENDS_DOC)
	$(XSLTPROC) \
	         --stringparam collect.xref.targets "only" \
	         --stringparam target.database.document "olinkdb_nochunk.xml" \
	         --stringparam targets.filename \
	                       "$(DOC_PATH)/isoknochunktargets.db" \
	         $(DOC_PATH)/isok_nochunk_cust.xsl \
	         $(DOC_PATH)/isok.xml

$(DOC_PATH)/agplnochunktargets.db: $(VALIDATE_PATH)/agpl-3.0 \
                     $(DOC_PATH)/isok_nochunk_cust.xsl \
                     $(DOC_PATH)/olinkdb_nochunk.xml \
                     $(MAKEDEPENDS_DOC)
	$(XSLTPROC) \
	         --stringparam collect.xref.targets "only" \
	         --stringparam target.database.document "olinkdb_nochunk.xml" \
	         --stringparam targets.filename \
	                       "$(DOC_PATH)/agplnochunktargets.db" \
	         $(DOC_PATH)/isok_nochunk_cust.xsl \
	         $(DOC_PATH)/agpl-3.0.dbk

$(DOC_PATH)/cc0nochunktargets.db: $(VALIDATE_PATH)/isok \
                     $(DOC_PATH)/isok_nochunk_cust.xsl \
                     $(DOC_PATH)/olinkdb_nochunk.xml \
                     $(MAKEDEPENDS_DOC)
	$(XSLTPROC) \
	         --stringparam collect.xref.targets "only" \
	         --stringparam target.database.document "olinkdb_nochunk.xml" \
	         --stringparam targets.filename \
	                       "$(DOC_PATH)/cc0nochunktargets.db" \
	         $(DOC_PATH)/isok_nochunk_cust.xsl \
	         $(DOC_PATH)/cc0-1.0-universal.xml
#
# Maintenance targets
#

.PHONY: maintainer-clean_docs
maintainer-clean_docs:
	rm -rf $(BUILT_PATH) \
	       $(VALIDATE_PATH) \
	       $(DOC_PATH)/isoktargets.db \
	       $(DOC_PATH)/isoknochunktargets.db \
	       $(DOC_PATH)/isokpdftargets.db \
	       $(DOC_PATH)/agpltargets.db \
	       $(DOC_PATH)/agplnochunktargets.db \
	       $(DOC_PATH)/agplpdftargets.db \
	       $(DOC_PATH)/cc0targets.db \
	       $(DOC_PATH)/cc0nochunktargets.db \
	       $(DOC_PATH)/cc0pdftargets.db \
	       $(DOC_PATH)/isok.fo \
	       $(DOC_PATH)/html_paginated


# Extend the functionality of the PGXS clean targets

.PHONY: maintainer-clean
maintainer-clean: maintainer-clean_docs
